<?php

namespace App\Http\Controllers\Api;

use Carbon\Carbon;
use App\Models\Email;
use App\Models\Saving;
use App\Models\Setting;
use App\Models\Customer;
use Illuminate\Http\Request;
use App\Traits\HttpResponses;
use App\Http\Traites\UserTraite;
use App\Http\Traites\SavingTraite;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use App\Http\Traites\LastloginTraite;
use App\Notifications\AppNotification;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Notification;

class LoginController extends Controller
{
    use LastloginTraite;
    use SavingTraite;
    use UserTraite;
    use HttpResponses;

    public function login(Request $r)
    {
        $this->logInfo("customer login", $r->except('password'));

        $validate = Validator::make($r->all(), [
            'username' => ['required', 'string'],
            'password' => ['required', 'string'],
        ]);

        if ($validate->fails()) {
            return $this->error(
                'Validation error.',
                422,
                $validate->errors()->toArray()
            );
        }

         $deviceid   = !empty($r->device_id) ? $r->device_id : $r->device_id;
        $devicetype = $deviceid == "web" ? "web" : "mobile";

         $getsetvalue = new Setting();

        if (Auth::guard('customer')->attempt(['username' => $r->username, 'password' => $r->password])) {

            $user = Customer::where('username', $r->username)->first();

            // Status checks
            if ($user->status == "6") {
                return $this->error('Your account has been blocked. Please contact support.', 401);
            } elseif ($user->status == "5") {
                return $this->error('Your account has been blocked due to a fraudulent attack. Please contact support or visit any of our branches.', 401);
            } elseif ($user->status == "4") {
                return $this->error('Your account has been restricted. Please contact support.', 401);
            } elseif ($user->status == "2") {
                return $this->error('Your account has been closed. Please contact support.', 401);
            } elseif ($user->status == "8") {
                return $this->error('Your account is dormant or inactive. Please contact support or visit any of our branches.', 401);
            } elseif ($user->status == "7") {
                return $this->error('Your account is currently being reviewed and will be approved soon.', 401);
            } elseif ($user->status == "9") {
                return $this->error('This account does not exist.', 401);
            }

            if (Auth::guard('customer')->user()->phone_verify == 1) {
                // reset failed counters
                $user->failed_logins  = null;
                $user->failed_balance = null;
                $user->failed_pin     = null;
                $user->last_login     = Carbon::now();
                $user->save();

                $savings = Saving::where('customer_id', Auth::guard('customer')->user()->id)->first();

                $userDetails = [
                    "userid"     => Auth::guard('customer')->user()->id,
                    "first_name" => Auth::guard('customer')->user()->first_name,
                    "last_name"  => Auth::guard('customer')->user()->last_name,
                    "phone"      => Auth::guard('customer')->user()->phone,
                    "profilepic" => url('/') . "/" . Auth::guard('customer')->user()->photo,
                    "username"   => Auth::guard('customer')->user()->username,
                    "bvn"        => Auth::guard('customer')->user()->bvn,
                    "nin"        => Auth::guard('customer')->user()->nin,
                    "email"      => Auth::guard('customer')->user()->email,
                    "address"    => Auth::guard('customer')->user()->address,
                    "sex"        => Auth::guard('customer')->user()->sex,
                    "accountno"  => Auth::guard('customer')->user()->acctno,
                    "balance"    => $savings ? $savings->account_balance : 0,
                    "currency"   => $getsetvalue->getsettingskey('currency_symbol'),
                ];

                $devcie = $this->CheckDeviceId($devicetype, $deviceid, Auth::guard('customer')->user()->id);

                $this->logInfo("device response", $devcie);

                if ($devcie["status"] == false) {
                    return response()->json($devcie, 406);
                }

                $dtoken = $this->CheckDeviceToken($r->device_token, Auth::guard('customer')->user()->id);

                $usertoken   = Auth::guard('customer')->user();

                if (method_exists($usertoken, 'tokens')) {
                    $usertoken->tokens()->delete();
                }

                $accessToken = $usertoken->createToken('customerToken', ['customer'], Carbon::now()->addMinutes(20));

                return $this->success('Login successfully.', [
                    'user'         => $userDetails,
                    'access_token' => $accessToken->plainTextToken,
                    'expired_at'   => date('Y-m-d H:i:s', strtotime($accessToken->accessToken->expires_at)),
                ]);
            } else {
                $otpCode = $this->generateSixOTP();
                $userauth                       = Customer::where('id', Auth::guard('customer')->user()->id)->first();
                $userauth->otp                  = $otpCode;
                $userauth->otp_expiration_date  = Carbon::now()->addMinutes(5);
                $userauth->save();

                $msg = "You requested an OTP: " . $otpCode . "<br> Do not share with anyone.";

                Email::create([
                    'user_id'   => Auth::guard('customer')->user()->id,
                    'subject'   => ucwords($getsetvalue->getsettingskey('company_name')) . " OTP Confirmation",
                    'message'   => $msg,
                    'recipient' => Auth::guard('customer')->user()->email,
                ]);


                Notification::route('mail', $userauth->email)
                    ->notify(new AppNotification('AssetMatrix MFB OTP Confirmation', $msg));


                return $this->error(
                    'Account pending verification.',
                    203,
                    [
                        'userid'      => Auth::guard('customer')->user()->id,
                        'email'       => Auth::guard('customer')->user()->email,
                    ]
                );
            }
        } else {
            // Auth failed
            $user = Customer::where('username', $r->username)->first();

            if ($user) {
                // make sure null doesn't break failed_logins
                $user->failed_logins = $user->failed_logins ?? 0;

                if ($user->failed_logins >= 3) {
                    $user->status = 6;
                    $user->save();

                    return $this->error(
                        'Your account has been blocked. Please contact admin to unlock your account.',
                        401
                    );
                } else {
                    $user->failed_logins += 1;
                    $user->save();

                    return $this->error(
                        'Invalid login credentials. Your account will be deactivated after ' . (4 - $user->failed_logins) . ' more attempt(s).',
                        401
                    );
                }
            } else {
                return $this->error('Invalid login credentials.', 401);
            }
        }
    }

    public function CheckDeviceId($devicetype, $deviceid, $userid)
    {
        $user = Customer::select('id', 'device_id', 'internet_banking')->where('id', $userid)->first();

        if ($devicetype == "web") {
            if ($user->internet_banking == 1) {
                return ['status' => true, 'message' => 'Internet banking enabled'];
            } else {
                return ['status' => false, 'message' => 'Internet banking is not enabled for this account, kindly contact support'];
            }
        } else {
            if (is_null($user->device_id)) {
                $uspdt            = Customer::where('id', $userid)->first();
                $uspdt->device_id = $deviceid . "_" . $user->id;
                $uspdt->save();

                return [
                    'status'   => true,
                    'message'  => 'Device linked',
                    'code'     => '104',
                    'deviceid' => $deviceid . "_" . $user->id,
                ];
            } else {
                $deeid = $deviceid . "_" . $user->id;

                if ($user->device_id == $deeid) {
                    return ['status' => true, 'message' => 'Device OK', 'code' => '104'];
                } else {
                    return [
                        'status'  => false,
                        'message' => 'Sorry, this device is not linked to this account',
                        'userid'  => $user->id,
                        'code'    => '105',
                    ];
                }
            }
        }
    }

    public function CheckDeviceToken($deviceToken, $userid)
    {
        $user = Customer::where('id', $userid)->first();

        if (is_null($user->device_token)) {
            $uspdt              = Customer::where('id', $userid)->first();
            $uspdt->device_token = $deviceToken;
            $uspdt->save();

            return [
                'status'      => true,
                'message'     => 'Device token linked',
                'code'        => '104',
                'deviceToken' => $deviceToken,
            ];
        } else {
            $deeid = $deviceToken;

            if ($user->device_token == $deeid) {
                return [
                    'status'      => true,
                    'message'     => 'Device token OK',
                    'deviceToken' => $user->device_token,
                    'code'        => '104',
                ];
            } else {
                return [
                    'status'  => false,
                    'message' => 'Sorry, this is an invalid device token',
                    'userid'  => $user->id,
                    'code'    => '105',
                ];
            }
        }
    }

    public function logout_customer()
    {
        Auth::user()->currentAccessToken()->delete();
        return $this->success('Logout successful.', null, '00', 201);
    }
}
